<!doctype html>
<html>
<head>
<meta charset="utf-8">
<title>Media Evaluation</title>
<link rel="stylesheet" href="assets/style.css">
<style>
/* Add these styles for logo and footer */
.logo-container {
    position: absolute;
    top: 20px;
    left: 20px;
}
.gsiq-logo {
    height: 40px;
    width: auto;
}
.footer {
    text-align: center;
    color: #666;
    font-size: 11px;
    margin-top: 20px;
    padding: 10px;
    border-top: 1px solid #333;
}

/* Improved multiselect styles */
.multiselect-container {
    position: relative;
    width: 200px;
}

.multiselect-header {
    padding: 8px 12px;
    border: 1px solid #ccc;
    background: white;
    cursor: pointer;
    border-radius: 4px;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.multiselect-dropdown {
    position: absolute;
    top: 100%;
    left: 0;
    right: 0;
    background: white;
    border: 1px solid #ccc;
    border-top: none;
    max-height: 200px;
    overflow-y: auto;
    display: none;
    z-index: 1000;
}

.multiselect-dropdown.show {
    display: block;
}

.multiselect-option {
    padding: 8px 12px;
    cursor: pointer;
    display: flex;
    align-items: center;
}

.multiselect-option:hover {
    background: #f0f0f0;
}

.multiselect-option input[type="checkbox"] {
    margin-right: 8px;
}

.multiselect-option.selected {
    background: #e8f4ff;
}

/* Filter row styling */
.filters {
    display: flex;
    gap: 10px;
    align-items: center;
    flex-wrap: wrap;
}

.filters select {
    padding: 8px 12px;
    border: 1px solid #ccc;
    border-radius: 4px;
    background: white;
    min-width: 150px;
}

.filters button {
    padding: 8px 16px;
    background: #007bff;
    color: white;
    border: none;
    border-radius: 4px;
    cursor: pointer;
}

.filters button:hover {
    background: #0056b3;
}

/* Selected chips for brands */
.selected-chips {
    display: flex;
    flex-wrap: wrap;
    gap: 5px;
    margin-top: 5px;
}

.chip {
    background: #e8f4ff;
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 12px;
    display: flex;
    align-items: center;
    gap: 4px;
}

.chip-remove {
    cursor: pointer;
    color: #666;
}

.chip-remove:hover {
    color: #ff0000;
}
</style>
</head>
<body>

<!-- Logo in top right corner -->
<div class="logo-container">
    <img src="https://gsiq.com/wp-content/uploads/2025/07/GSIQ-100px.png" alt="GSIQ Logo" class="gsiq-logo">
</div>

<div class="page">
  <div class="header">
    <div>
      <div class="h-title">Media Evaluation</div>
      <div class="h-sub">Brand filter (single or multiple)</div>
    </div>

    <div class="filters">
      <select id="project"></select>
      <select id="episode"></select>
      
      <!-- Custom multiselect for brands -->
      <div class="multiselect-container">
        <div class="multiselect-header" onclick="toggleBrandsDropdown()">
          <span id="brands-placeholder">Select Brands</span>
          <span>▼</span>
        </div>
        <div class="multiselect-dropdown" id="brands-dropdown">
          <!-- Brands will be populated here -->
        </div>
        <div class="selected-chips" id="selected-chips">
          <!-- Selected brands will appear as chips -->
        </div>
      </div>
      
      <button onclick="loadData()">Apply</button>
    </div>
  </div>

  <!-- MINIMAL: Selected brands line (no layout changes) -->
  <div id="selectedBrands" style="margin:12px 0 10px 0;color:#aaa;font-size:16px;"></div>

  <table id="tbl">
    <thead>
      <tr>
        <th class="c-asset">ASSETS</th>
        <th class="c-right">DURATION</th>
        <th class="c-right">SHARE EXP</th>
        <th class="c-right">GROSS</th>
        <th class="c-right">NET</th>
        <th class="c-right">SOV</th>
        <th class="c-right">AIS %</th>
      </tr>
    </thead>
    <tbody></tbody>
    <tfoot></tfoot>
  </table>

  <!-- Footer -->
  <div class="footer">
    Copyright (C) 2026, GSIQ, All rights reserved
  </div>
</div>

<script>
// CHANGED: Removed the "? " prefix
function inr(n){ return Number(n).toLocaleString("en-IN"); }

// Store selected brands globally
let selectedBrands = new Set();

function toggleBrandsDropdown() {
    const dropdown = document.getElementById('brands-dropdown');
    dropdown.classList.toggle('show');
}

function closeBrandsDropdown() {
    const dropdown = document.getElementById('brands-dropdown');
    dropdown.classList.remove('show');
}

function toggleBrand(brand) {
    if (selectedBrands.has(brand)) {
        selectedBrands.delete(brand);
    } else {
        selectedBrands.add(brand);
    }
    updateBrandsDisplay();
    loadData(); // Auto-refresh when brands change
}

function removeBrand(brand) {
    selectedBrands.delete(brand);
    updateBrandsDisplay();
    loadData(); // Auto-refresh when brands change
}

function updateBrandsDisplay() {
    const placeholder = document.getElementById('brands-placeholder');
    const chipsContainer = document.getElementById('selected-chips');
    const dropdown = document.getElementById('brands-dropdown');
    
    // Update placeholder
    if (selectedBrands.size === 0) {
        placeholder.textContent = 'Select Brands';
    } else if (selectedBrands.size === 1) {
        placeholder.textContent = Array.from(selectedBrands)[0];
    } else {
        placeholder.textContent = `${selectedBrands.size} brands selected`;
    }
    
    // Update chips
    chipsContainer.innerHTML = '';
    selectedBrands.forEach(brand => {
        const chip = document.createElement('div');
        chip.className = 'chip';
        chip.innerHTML = `
            ${brand}
            <span class="chip-remove" onclick="removeBrand('${brand}')">×</span>
        `;
        chipsContainer.appendChild(chip);
    });
    
    // Update checkboxes in dropdown
    const checkboxes = dropdown.querySelectorAll('input[type="checkbox"]');
    checkboxes.forEach(checkbox => {
        checkbox.checked = selectedBrands.has(checkbox.value);
        const option = checkbox.closest('.multiselect-option');
        if (checkbox.checked) {
            option.classList.add('selected');
        } else {
            option.classList.remove('selected');
        }
    });
}

async function loadFilters(p="", e=""){
  const r = await fetch(`api.php?action=filters&project=${encodeURIComponent(p)}`);
  const j = await r.json();

  const ps=document.getElementById("project");
  const es=document.getElementById("episode");
  const bsDropdown=document.getElementById("brands-dropdown");

  // Clear and populate projects
  ps.innerHTML='<option value="">All Projects</option>';
  j.projects.forEach(x=>ps.innerHTML+=`<option value="${x}">${x}</option>`);

  // Clear and populate episodes
  es.innerHTML='<option value="">All Episodes</option>';
  j.episodes.forEach(x=>es.innerHTML+=`<option value="${x}">${x}</option>`);

  // Clear and populate brands dropdown
  bsDropdown.innerHTML='';
  j.brands.forEach(brand=>{
    const option = document.createElement('div');
    option.className = 'multiselect-option';
    option.innerHTML = `
      <input type="checkbox" value="${brand}" 
             onchange="toggleBrand('${brand}')" 
             ${selectedBrands.has(brand) ? 'checked' : ''}>
      ${brand}
    `;
    if (selectedBrands.has(brand)) {
      option.classList.add('selected');
    }
    bsDropdown.appendChild(option);
  });

  if(p) ps.value=p;
  if(e) es.value=e;
}

async function loadData(){
  const p=document.getElementById("project").value;
  const e=document.getElementById("episode").value;
  const bs=Array.from(selectedBrands);

  // MINIMAL: show selected brands above the table
  const sb = document.getElementById("selectedBrands");
  const brandText = bs.length ? bs.join(", ") : "All Brands";
  sb.innerText = bs.length
    ? (bs.length > 1 ? "Brands: " : "Brand: ") + brandText
    : "Brands: All Brands";
  sb.title = brandText; // hover shows full list

  const params = new URLSearchParams();
  if(p) params.append("project",p);
  if(e) params.append("episode",e);
  bs.forEach(b=>params.append("brands[]",b));

  const r=await fetch(`api.php?${params.toString()}`);
  const j=await r.json();

  const tb=document.querySelector("#tbl tbody");
  const tf=document.querySelector("#tbl tfoot");
  tb.innerHTML="";
  tf.innerHTML="";

  j.rows.forEach(r=>{
    tb.innerHTML+=`
      <tr>
        <td class="c-asset">${r.asset}</td>
        <td class="c-right">${r.duration}</td>
        <td class="c-right">${r.share_exposure}%</td>
        <td class="c-right">${inr(r.gross)}</td>
        <td class="c-right">${inr(r.net)}</td>
        <td class="c-right">${r.share_voice}%</td>
        <td class="c-right">${r.ais}%</td>
      </tr>`;
  });

  tf.innerHTML=`
    <tr class="total">
      <td class="c-asset">TOTAL</td>
      <td class="c-right">${j.totals.duration}</td>
      <td class="c-right"></td>
      <td class="c-right">${inr(j.totals.gross)}</td>
      <td class="c-right">${inr(j.totals.net)}</td>
      <td class="c-right"></td>
      <td class="c-right">${j.totals.ais}%</td>
    </tr>`;
}

// Auto-sync functionality
async function handleProjectChange() {
  const project = document.getElementById("project").value;
  await loadFilters(project, "");
  await loadData();
}

async function handleEpisodeChange() {
  await loadData();
}

// Close dropdown when clicking outside
document.addEventListener('click', function(event) {
  const brandsContainer = document.querySelector('.multiselect-container');
  if (!brandsContainer.contains(event.target)) {
    closeBrandsDropdown();
  }
});

// Initialize with event listeners
(async()=>{
  await loadFilters();
  await loadData();
  
  // Add event listeners for auto-sync
  document.getElementById("project").addEventListener("change", handleProjectChange);
  document.getElementById("episode").addEventListener("change", handleEpisodeChange);
})();
</script>
</body>
</html>
