<?php
header('Content-Type: application/json; charset=utf-8');

$DB_INI   = __DIR__ . '/db.ini';
$RATE_INI = __DIR__ . '/rate.ini';
$TABLE    = 'exposure_raw';

/* ---------------- Helpers ---------------- */
function respond($arr) {
  echo json_encode($arr, JSON_UNESCAPED_UNICODE);
  exit;
}

function norm_key($s) {
  $s = trim((string)$s);
  $s = preg_replace('/\s+/', ' ', $s);

  // Convert bracket suffix to _A/_P for easier matching
  $s = preg_replace('/\(\s*A\s*\)$/i', '_A', $s);
  $s = preg_replace('/\(\s*P\s*\)$/i', '_P', $s);

  // Normalize separators
  $s = str_replace([' - ', '-', '/', '\\'], ' ', $s);

  // Anything non-alnum => underscore
  $s = preg_replace('/[^A-Za-z0-9]+/', '_', $s);
  $s = preg_replace('/_+/', '_', $s);
  $s = trim($s, '_');
  return $s;
}

function project_candidates($p) {
  $p = trim((string)$p);
  $c = [];
  if ($p !== '') $c[] = $p;

  // Strip suffixes like (TV) / (OTT)
  $base = preg_replace('/\s*\((TV|OTT)\)\s*$/i', '', $p);
  $base = trim($base);
  if ($base !== '' && $base !== $p) $c[] = $base;

  // Unique
  $c = array_values(array_unique($c));
  return $c;
}

function ini_load($path) {
  if (!file_exists($path)) return [];
  $ini = parse_ini_file($path, true, INI_SCANNER_RAW);
  return is_array($ini) ? $ini : [];
}

function ini_get_rate($ini, $project) {
  foreach (project_candidates($project) as $p) {
    if (isset($ini[$p]['rate']) && $ini[$p]['rate'] !== '') return floatval($ini[$p]['rate']);
  }
  if (isset($ini['DEFAULT']['rate']) && $ini['DEFAULT']['rate'] !== '') return floatval($ini['DEFAULT']['rate']);
  return 0.0;
}

function ini_get_value_any($ini, $project, $keyExact, $keyNorm) {
  foreach (project_candidates($project) as $p) {
    if (!isset($ini[$p])) continue;
    if (isset($ini[$p][$keyExact]) && $ini[$p][$keyExact] !== '') return floatval($ini[$p][$keyExact]);
    if (isset($ini[$p][$keyNorm])  && $ini[$p][$keyNorm]  !== '') return floatval($ini[$p][$keyNorm]);
  }
  if (isset($ini['DEFAULT'])) {
    if (isset($ini['DEFAULT'][$keyExact]) && $ini['DEFAULT'][$keyExact] !== '') return floatval($ini['DEFAULT'][$keyExact]);
    if (isset($ini['DEFAULT'][$keyNorm])  && $ini['DEFAULT'][$keyNorm]  !== '') return floatval($ini['DEFAULT'][$keyNorm]);
  }
  return null;
}

function ini_get_brand_mult($ini, $project, $brand) {
  $brand = trim((string)$brand);
  if ($brand === '') return 1.0;

  // Accept BOTH:
  // brand.Fortune Maida  (exact)
  // brand.Fortune_Maida  (normalized)
  $exact = 'brand.' . $brand;
  $norm  = 'brand.' . norm_key($brand);

  $v = ini_get_value_any($ini, $project, $exact, $norm);
  return ($v === null) ? 1.0 : $v;
}

function ini_get_asset_mult($ini, $project, $asset) {
  $asset = trim((string)$asset);
  if ($asset === '') return 1.0;

  // Try exact + normalized first
  $exact = 'asset.' . $asset;
  $norm  = 'asset.' . norm_key($asset);

  $v = ini_get_value_any($ini, $project, $exact, $norm);
  if ($v !== null) return $v;

  // If not found, apply (A)=1.4 base asset when only base exists
  $isA = (bool)preg_match('/\(\s*A\s*\)$/i', $asset) || (bool)preg_match('/_A$/i', norm_key($asset));
  $isP = (bool)preg_match('/\(\s*P\s*\)$/i', $asset) || (bool)preg_match('/_P$/i', norm_key($asset));

  if ($isA || $isP) {
    // Strip suffix to derive base
    $base = preg_replace('/\s*\(\s*[AP]\s*\)\s*$/i', '', $asset);
    $base = preg_replace('/_([AP])$/i', '', norm_key($base));
    $base = trim($base);

    if ($base !== '') {
      $baseExact = 'asset.' . $base;          // might already be underscore-ish
      $baseNorm  = 'asset.' . norm_key($base);

      $bv = ini_get_value_any($ini, $project, $baseExact, $baseNorm);
      if ($bv !== null) {
        return $isA ? ($bv * 1.4) : $bv;
      }
    }
  }

  return 1.0;
}

function seconds_to_hms($secs) {
  $secs = max(0, (int)$secs);
  $h = intdiv($secs, 3600);
  $m = intdiv($secs % 3600, 60);
  $s = $secs % 60;
  return sprintf("%d:%02d:%02d", $h, $m, $s);
}

/* ---------------- DB ---------------- */
$db = @parse_ini_file($DB_INI);
if (!$db || !isset($db['host'],$db['user'],$db['password'],$db['database'])) {
  respond(['error' => 'db.ini missing/invalid']);
}

$mysqli = @new mysqli($db['host'], $db['user'], $db['password'], $db['database']);
if ($mysqli->connect_errno) respond(['error' => 'DB connect failed: ' . $mysqli->connect_error]);
$mysqli->set_charset('utf8mb4');

/* ---------------- INI ---------------- */
$ini = ini_load($RATE_INI);

/* ---------------- Filters endpoint ---------------- */
$action = $_GET['action'] ?? '';
if ($action === 'filters') {
  $project = trim($_GET['project'] ?? '');

  $projects = [];
  $res = $mysqli->query("SELECT DISTINCT project_name FROM `$TABLE` WHERE project_name IS NOT NULL AND project_name<>'' ORDER BY project_name");
  while ($res && ($row = $res->fetch_assoc())) $projects[] = $row['project_name'];

  $episodes = [];
  if ($project !== '') {
    $stmt = $mysqli->prepare("SELECT DISTINCT match_episode FROM `$TABLE` WHERE project_name=? AND match_episode IS NOT NULL AND match_episode<>'' ORDER BY match_episode");
    $stmt->bind_param("s", $project);
  } else {
    $stmt = $mysqli->prepare("SELECT DISTINCT match_episode FROM `$TABLE` WHERE match_episode IS NOT NULL AND match_episode<>'' ORDER BY match_episode");
  }
  $stmt->execute();
  $r = $stmt->get_result();
  while ($r && ($row = $r->fetch_assoc())) $episodes[] = $row['match_episode'];
  $stmt->close();

  $brands = [];
  if ($project !== '') {
    $stmt = $mysqli->prepare("SELECT DISTINCT brand FROM `$TABLE` WHERE project_name=? AND brand IS NOT NULL AND brand<>'' ORDER BY brand");
    $stmt->bind_param("s", $project);
  } else {
    $stmt = $mysqli->prepare("SELECT DISTINCT brand FROM `$TABLE` WHERE brand IS NOT NULL AND brand<>'' ORDER BY brand");
  }
  $stmt->execute();
  $r = $stmt->get_result();
  while ($r && ($row = $r->fetch_assoc())) $brands[] = $row['brand'];
  $stmt->close();

  respond(['projects'=>$projects, 'episodes'=>$episodes, 'brands'=>$brands]);
}

/* ---------------- Data endpoint ---------------- */
$project = trim($_GET['project'] ?? '');
$episode = trim($_GET['episode'] ?? '');
$brands  = $_GET['brands'] ?? [];
if (!is_array($brands)) $brands = [];
$brands = array_values(array_filter(array_map('trim', $brands), fn($x)=>$x!==''));

$where = [];
$params = [];
$types  = '';

if ($project !== '') { $where[] = "project_name=?"; $params[] = $project; $types .= 's'; }
if ($episode !== '') { $where[] = "match_episode=?"; $params[] = $episode; $types .= 's'; }

if (count($brands) > 0) {
  $in = implode(',', array_fill(0, count($brands), '?'));
  $where[] = "brand IN ($in)";
  foreach ($brands as $b) { $params[] = $b; $types .= 's'; }
}

$whereSql = count($where) ? ('WHERE ' . implode(' AND ', $where)) : '';

/* total count for share calculations */
$stmt = $mysqli->prepare("SELECT COUNT(*) AS total_cnt FROM `$TABLE` $whereSql");
if ($types !== '') $stmt->bind_param($types, ...$params);
$stmt->execute();
$totalRes = $stmt->get_result()->fetch_assoc();
$totalCnt = (int)($totalRes['total_cnt'] ?? 0);
$stmt->close();

if ($totalCnt <= 0) respond(['rows'=>[], 'totals'=>['duration'=>'0:00:00','gross'=>0,'net'=>0,'ais'=>0]]);

/* Compute blended brand multiplier if multiple brands selected */
$brandMult = 1.0;
if (count($brands) === 1) {
  $brandMult = ini_get_brand_mult($ini, $project, $brands[0]);
} elseif (count($brands) > 1) {
  // Weighted average by actual counts within current filter
  $in = implode(',', array_fill(0, count($brands), '?'));
  $sql = "SELECT brand, COUNT(*) AS cnt FROM `$TABLE` $whereSql GROUP BY brand";
  $stmt = $mysqli->prepare($sql);
  if ($types !== '') $stmt->bind_param($types, ...$params);
  $stmt->execute();
  $r = $stmt->get_result();

  $sumCnt = 0;
  $sumW   = 0.0;
  while ($row = $r->fetch_assoc()) {
    $b = (string)$row['brand'];
    $c = (int)$row['cnt'];
    $w = ini_get_brand_mult($ini, $project, $b);
    $sumCnt += $c;
    $sumW   += ($w * $c);
  }
  $stmt->close();
  $brandMult = ($sumCnt > 0) ? ($sumW / $sumCnt) : 1.0;
}

/* Rate */
$rate = ini_get_rate($ini, $project);

/* Group by asset/location */
$stmt = $mysqli->prepare("SELECT location AS asset, COUNT(*) AS cnt
                          FROM `$TABLE`
                          $whereSql
                          GROUP BY location
                          ORDER BY cnt DESC, location ASC");
if ($types !== '') $stmt->bind_param($types, ...$params);
$stmt->execute();
$res = $stmt->get_result();

$rows = [];
$totalGross = 0.0;
$totalNet   = 0.0;
$totalSecs  = 0;

while ($row = $res->fetch_assoc()) {
  $asset = (string)$row['asset'];
  $cnt   = (int)$row['cnt'];

  $secs = $cnt; // 1 instance = 1 second
  $totalSecs += $secs;

  $shareExp = ($totalCnt > 0) ? ($cnt * 100.0 / $totalCnt) : 0.0;

  $assetMult = ini_get_asset_mult($ini, $project, $asset);

  $gross = $rate * $cnt;
  $net   = $gross * (($brandMult + $assetMult) / 2);

  $totalGross += $gross;
  $totalNet   += $net;

  $aisPct = ($gross > 0) ? (($net / $gross) * 100.0) : 0.0;

  $rows[] = [
    'asset' => $asset,
    'duration' => seconds_to_hms($secs),
    'share_exposure' => round($shareExp, 1),
    'gross' => round($gross, 0),
    'net'   => round($net, 0),
    'share_voice' => round($shareExp, 1),
    'ais' => round($aisPct, 1),
  ];
}
$stmt->close();

$totAis = ($totalGross > 0) ? (($totalNet / $totalGross) * 100.0) : 0.0;

respond([
  'rows' => $rows,
  'totals' => [
    'duration' => seconds_to_hms($totalSecs),
    'gross' => round($totalGross, 0),
    'net'   => round($totalNet, 0),
    'ais'   => round($totAis, 1),
  ],
]);
