<?php /* /var/www/html/bls/dashboard.php */ ?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <title>Impact of Sponsorship – BLS</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />

  <!-- Styles (your file) -->
  <link rel="stylesheet" href="./assets/style.css?v=5" />
  <link rel="stylesheet" href="./assets/chat.css?v=1">

  <!-- Chart.js -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.6/dist/chart.umd.min.js"></script>
  <script src="./assets/chat.js?v=1"></script>

  <!-- PDF export -->
  <script src="https://cdn.jsdelivr.net/npm/jspdf@2.5.1/dist/jspdf.umd.min.js"></script>
</head>
<body>
<div class="wrapper">

  <!-- print header/footer (hidden on screen) -->
  <div class="print-header">
    <div><strong>IMPACT OF SPONSORSHIP</strong> — <span id="phBrand">Brand</span> (<span id="phEvent">Event</span>)</div>
    <div id="phDate"></div>
  </div>
  <div class="print-footer">
    <div>© GSIQ • Generated from bls_scores</div>
    <div class="page-num"></div>
  </div>

  <!-- Header with inline logo + title (ONLY UI change) -->
  <div class="header">
    <div>
      <div style="display:flex; align-items:center; gap:12px; flex-wrap:wrap;">
        <img src="https://gsiq.com/wp-content/uploads/2025/07/GSIQ-100px.png"
             alt="GSIQ" style="height:52px;width:auto">
        <h1 class="h-title" style="margin:0;">IMPACT OF SPONSORSHIP</h1>
      </div>
      <div class="h-sub">Primary Research: Controlled, Exposed-Pre and Exposed-Post</div>
      <!-- Dynamic Sample Size / Target Group -->
      <div class="small" id="sampleTarget"></div>
    </div>
    <div class="small badge" id="badgeEvent">Event</div>
  </div>

  <div class="controls">
    <label>
      Event
      <select id="eventSelect"></select>
    </label>
    <label>
      Brand
      <select id="brandSelect"></select>
    </label>
    <button class="primary" id="refreshBtn">Refresh</button>
    <button id="downloadCsvBtn">Download CSV</button>
    <button id="exportPngBtn">Export PNG</button>
    <button id="exportPdfBtn">Export PDF</button>
    <button id="printBtn">Print</button>
  </div>

  <!-- Chart 1 -->
  <section class="card avoid-break-inside">
    <h3>Control vs Exposed (Pre &amp; Post) Comparison</h3>
    <div class="chart-wrap"><canvas id="chartGrouped" class="canvas"></canvas></div>
    <div class="legend-note">Series: Control, Exposed_Pre, Exposed_Post</div>
  </section>

  <!-- Chart 2 -->
  <section class="card avoid-break-inside">
    <h3>Funnel – Exposed_Post</h3>
    <div class="chart-wrap"><canvas id="chartFunnel" class="canvas funnel"></canvas></div>
  </section>

  <!-- Chart 3 -->
  <section class="card avoid-break-inside">
    <h3>Brand Lift from Exposure – Delta (Post vs Pre)</h3>
    <div class="chart-wrap"><canvas id="chartDelta" class="canvas delta"></canvas></div>
  </section>

  <div class="footer-note screen-only">
    This dashboard maps the brand impact journey and lets you compare Control → Pre → Post, inspect the exposed funnel,
    and quantify delta lifts. Values are percentages (0–100).
  </div>
</div>

<script>
/* ---------- setup ---------- */
const eventSel    = document.getElementById('eventSelect');
const brandSel    = document.getElementById('brandSelect');
const refreshBtn  = document.getElementById('refreshBtn');
const downloadBtn = document.getElementById('downloadCsvBtn');
const exportPngBtn= document.getElementById('exportPngBtn');
const exportPdfBtn= document.getElementById('exportPdfBtn');
const printBtn    = document.getElementById('printBtn');
const phBrand     = document.getElementById('phBrand');
const phDate      = document.getElementById('phDate');
const phEvent     = document.getElementById('phEvent');
const badgeEvent  = document.getElementById('badgeEvent');
const sampleTargetEl = document.getElementById('sampleTarget');

let chartGrouped, chartFunnel, chartDelta;

const c1 = 'rgba(128,137,155,0.75)'; // Control
const c2 = 'rgba(74, 102, 203,0.85)';// Pre
const c3 = 'rgba(40, 72, 201,0.95)'; // Post
const axisColor = '#6b7280';

/* mobile helpers */
const MOBILE_Q = window.matchMedia('(max-width: 768px)');
function isMobile(){ return MOBILE_Q.matches; }
MOBILE_Q.addEventListener?.('change', ()=> { loadCharts(); });

async function fetchJSON(url){ const r = await fetch(url, {cache:'no-store'}); return r.json(); }
function getParam(name){ const u = new URL(window.location.href); return u.searchParams.get(name); }
function setParam(name, value){ const u = new URL(window.location.href); u.searchParams.set(name, value); history.replaceState(null, '', u); }

function tidyRows(rows){
  // Generic preferred order for BLS metrics; anything else falls to the end
  const order = [
    'Unaided Brand Recall (Finance)', 'Unaided Brand Recall (Insurance)', 'Unaided Brand Recall (Beverage)', 'Unaided Brand Recall (Appliance)',
    'Aided Brand Recall',
    'Top Finance Brand in Series', 'Top Insurance Brand in Series', 'Top Beverage Brand in Series', 'Top Appliance Brand in Series',
    'Saw Ad or Mention During Series',
    'Favorable Opinion (4–5)',
    'Would Consider / Open Account', 'Would Consider Policy', 'Would Consider / Try', 'Would Consider / Buy',
    'Likely to Register',
    'Actually enrolled During Series', 'Actually Purchased During Series',
    'Brand Fit with Series',
    'Series Improved View of IDFC First Bank', 'Series Improved View of SBI Life', 'Series Improved View of Campa', 'Series Improved View of Atomberg'
  ];
  const lc = s => String(s||'').toLowerCase();
  const pos = m => { const i = order.findIndex(x => lc(x) === lc(m)); return i === -1 ? 999 : i; };
  return [...rows].sort((a,b)=> pos(a.Metric)-pos(b.Metric));
}
function num(v){
  if (v === null || v === undefined) return null;
  const n = parseFloat(String(v).replace(/,/g,'').replace('%',''));
  return Number.isFinite(n) ? n : null;
}
function wrapLabel(s, max=18){
  const words = String(s).split(' '), lines = []; let line = '';
  for (const w of words){ if ((line+' '+w).trim().length>max){ lines.push(line.trim()); line=w; } else { line=(line?line+' ':'')+w; } }
  if (line) lines.push(line.trim()); return lines;
}
function normalizeToPercent(seriesList){
  const all = seriesList.flat().filter(v => v !== null && !isNaN(v));
  const max = all.length ? Math.max(...all) : 0;
  const factor = (max <= 1.5 ? 100 : 1);
  return { factor, series: seriesList.map(a => a.map(v => v==null ? null : +(v*factor).toFixed(2))) };
}
function mobileify(baseOpts){
  if (!isMobile()) return baseOpts;
  const b = JSON.parse(JSON.stringify(baseOpts));
  if (b.scales?.x?.ticks) b.scales.x.ticks.font = { size: 10 };
  if (b.scales?.y?.ticks) b.scales.y.ticks.font = { size: 10 };
  if (b.plugins?.legend?.labels){
    b.plugins.legend.labels.boxWidth = 12;
    b.plugins.legend.labels.font = { size: 10, weight: 700 };
  }
  if (b.layout?.padding?.top) b.layout.padding.top = Math.max(16, b.layout.padding.top - 6);
  return b;
}

/* ---------- custom value labels plugin ---------- */
const valueLabels = {
  id: 'valueLabels',
  afterDatasetsDraw(chart){
    const { ctx, data, scales } = chart;
    const isVerticalChart = chart.options.indexAxis !== 'y';
    const isGrouped = chart.canvas.id === 'chartGrouped';

    ctx.save();
    ctx.font = (isMobile() ? '700 10px Montserrat, Arial' : '700 11px Montserrat, Arial');
    ctx.textBaseline = 'middle';

    const fmt = v => {
      const s = Number(v).toFixed(2);
      return (s.endsWith('.00') ? s.slice(0,-3) : s) + '%';
    };

    data.datasets.forEach((ds, di) => {
      const meta = chart.getDatasetMeta(di);
      if (meta.hidden) return;

      meta.data.forEach((elem, i) => {
        const raw = ds.data[i];
        if (raw == null || isNaN(raw)) return;

        if (isVerticalChart) {
          const x = elem.x;
          const yTop = scales.y.getPixelForValue(raw);

          if (isGrouped) {
            const y = yTop - 6;
            ctx.save();
            ctx.translate(x, y);
            ctx.rotate(-Math.PI / 2);
            ctx.fillStyle = '#1f2937';
            ctx.textAlign = 'left';
            ctx.fillText(fmt(raw), 0, 0);
            ctx.restore();
          } else {
            const putInside = raw >= 15;
            ctx.fillStyle = putInside ? '#ffffff' : '#1f2937';
            const y = putInside ? (yTop + 12) : (yTop - 8);
            ctx.textAlign = 'center';
            ctx.fillText(fmt(raw), x, y);
          }

        } else {
          const y = elem.y;
          const xVal = scales.x.getPixelForValue(raw);
          const putInside = raw >= 15;
          ctx.textAlign = putInside ? 'right' : 'left';
          ctx.fillStyle = putInside ? '#ffffff' : '#1f2937';
          const x = putInside ? (xVal - 6) : (xVal + 6);
          ctx.fillText(fmt(raw), x, y);
        }
      });
    });
    ctx.restore();
  }
};
Chart.register(valueLabels);

/* ---------- chart builders ---------- */
function makeGrouped(ctx, labels, control, pre, post){
  const opts = mobileify({
    responsive:true,
    layout:{ padding:{ top: 50 } },
    scales:{
      x:{ticks:{color:axisColor, maxRotation:30, autoSkip:true}},
      y:{ticks:{color:axisColor, callback:(v)=> v+'%'}, beginAtZero:true, suggestedMax:100, grid:{color:'#eef2f7'}}
    },
    plugins:{
      legend:{ labels:{ color: axisColor, font:{ weight:700 } } },
      tooltip:{ callbacks:{ label:c => `${c.dataset.label?c.dataset.label+': ':''}${c.parsed.y}%` } }
    }
  });
  const commonDS = isMobile() ? { maxBarThickness: 22, barPercentage: 0.85, categoryPercentage: 0.72 } : {};
  return new Chart(ctx, {
    type:'bar',
    data:{ labels, datasets:[
      {label:'Control',       data:control, backgroundColor:c1, ...commonDS},
      {label:'Exposed_Pre',   data:pre,     backgroundColor:c2, ...commonDS},
      {label:'Exposed_Post',  data:post,    backgroundColor:c3, ...commonDS},
    ]},
    options: opts,
    plugins:[valueLabels]
  });
}

function makeFunnel(ctx, labels, post){
  const opts = mobileify({
    indexAxis:'y',
    layout:{ padding:{ right: 20 } },
    scales:{
      x:{ticks:{color:axisColor, callback:(v)=> v+'%'}, beginAtZero:true, suggestedMax:100, grid:{color:'#eef2f7'}},
      y:{ticks:{color:axisColor}}
    },
    plugins:{ legend:{ display:false }, tooltip:{ callbacks:{ label:c => `${c.parsed.x}%` } } }
  });
  const commonDS = isMobile() ? { barPercentage: 0.8, categoryPercentage: 0.8, maxBarThickness: 26 } : {};
  return new Chart(ctx, {
    type:'bar',
    data:{ labels, datasets:[{label:'Exposed_Post', data:post, backgroundColor:c3, ...commonDS}]},
    options: opts,
    plugins:[valueLabels]
  });
}

function makeDelta(ctx, labels, delta){
  const opts = mobileify({
    layout:{ padding:{ top: 24 } },
    scales:{
      x:{ticks:{color:axisColor}},
      y:{ticks:{color:axisColor, callback:(v)=> v+'%'}, beginAtZero:true, grid:{color:'#eef2f7'}}
    },
    plugins:{ legend:{ display:false }, tooltip:{ callbacks:{ label:c => `${c.parsed.y>=0?'+':''}${c.parsed.y}%` } } }
  });
  const commonDS = isMobile() ? { maxBarThickness: 22, barPercentage: 0.85, categoryPercentage: 0.72 } : {};
  return new Chart(ctx, {
    type:'bar',
    data:{ labels, datasets:[{label:'Post - Pre', data:delta, backgroundColor:c3, ...commonDS}]},
    options: opts,
    plugins:[valueLabels]
  });
}

/* ---------- load + render ---------- */
async function loadEvents(){
  const events = await fetchJSON('./api.php?action=events');
  const want = getParam('event');
  eventSel.innerHTML = events.map(e=>`<option>${e}</option>`).join('');
  if (want && events.includes(want)) eventSel.value = want;
  setParam('event', eventSel.value);
  badgeEvent.textContent = eventSel.value || 'Event';
  phEvent.textContent = eventSel.value || 'Event';
}

async function loadBrands(){
  const event = eventSel.value || getParam('event') || '';
  const brands = await fetchJSON('./api.php?action=brands&event='+encodeURIComponent(event));
  const want = getParam('brand');
  brandSel.innerHTML = brands.map(b=>`<option>${b}</option>`).join('');
  if (want && brands.includes(want)) brandSel.value = want;
  setParam('brand', brandSel.value);
}

async function loadCharts(){
  const event = eventSel.value || getParam('event') || '';
  const brand = brandSel.value;

  setParam('event', event);
  setParam('brand', brand);

  badgeEvent.textContent = event;
  phEvent.textContent = event;
  phBrand.textContent = brand;
  phDate.textContent  = new Date().toLocaleString();

  // reset sample/target then set from API meta
  if (sampleTargetEl) sampleTargetEl.textContent = '';

  const res  = await fetchJSON('./api.php?action=data&brand='+encodeURIComponent(brand)+'&event='+encodeURIComponent(event));
  const rows = tidyRows(res.rows || []);

  const meta = res.meta || {};
  if (sampleTargetEl && (meta.Sample_Size || meta.Target_Group)) {
    let t = '';
    if (meta.Sample_Size)  t += 'n=' + meta.Sample_Size;
    if (meta.Target_Group) t += (t ? ', ' : '') + meta.Target_Group;
    sampleTargetEl.textContent = t;
  }

  const rawLabels = rows.map(r=> r.Metric);
  const controlRaw = rows.map(r=> num(r.Control));
  const preRaw     = rows.map(r=> num(r.Exposed_Pre));
  const postRaw    = rows.map(r=> num(r.Exposed_Post));

  const norm = normalizeToPercent([controlRaw, preRaw, postRaw]);
  const control = norm.series[0], pre = norm.series[1], post = norm.series[2];

  const delta = post.map((p,i)=> (p==null || pre[i]==null) ? null : +(p - pre[i]).toFixed(2));
  const labels = rawLabels.map(l => wrapLabel(l, isMobile() ? 14 : 18));

  chartGrouped && chartGrouped.destroy();
  chartFunnel  && chartFunnel.destroy();
  chartDelta   && chartDelta.destroy();

  chartGrouped = makeGrouped(document.getElementById('chartGrouped'), labels, control, pre, post);

  const idx = post.map((v,i)=>({v,i})).sort((a,b)=> (b.v??-1) - (a.v??-1)).map(o=>o.i);
  chartFunnel = makeFunnel(
    document.getElementById('chartFunnel'),
    idx.map(i=> labels[i]),
    idx.map(i=> post[i])
  );

  chartDelta = makeDelta(document.getElementById('chartDelta'), labels, delta);
}

/* ---------- exports ---------- */
function exportPNG(){
  const brand = brandSel.value;
  const event = eventSel.value;
  const canvases = [
    document.getElementById('chartGrouped'),
    document.getElementById('chartFunnel'),
    document.getElementById('chartDelta')
  ];
  const pad = 20, gap = 30, headerH = 70;
  const width  = Math.max(...canvases.map(c=>c.width)) + pad*2;
  let totalH = headerH + pad + canvases.reduce((h,c)=> h + Math.round((c.height/c.width)*(width-pad*2)) + gap, 0);
  const off = document.createElement('canvas');
  off.width = width; off.height = totalH;
  const ctx = off.getContext('2d');
  ctx.fillStyle = '#ffffff'; ctx.fillRect(0,0,off.width,off.height);
  ctx.fillStyle = '#2b2f42'; ctx.font = '700 24px Montserrat, Arial';
  ctx.fillText(`Impact of Sponsorship — ${brand}`, pad, 38);
  ctx.font = '500 12px Montserrat, Arial';
  ctx.fillText(`${event} • Generated from bls_scores`, pad, 58);

  let y = headerH;
  canvases.forEach(c=>{
    const w = width - pad*2;
    const h = Math.round((c.height / c.width) * w);
    ctx.drawImage(c, pad, y, w, h);
    y += h + gap;
  });

  const a = document.createElement('a');
  a.href = off.toDataURL('image/png', 1.0);
  a.download = `bls_${event}_${brand}.png`;
  a.click();
}
async function exportPDF(){
  const { jsPDF } = window.jspdf;
  const brand = brandSel.value;
  const event = eventSel.value;
  const pdf = new jsPDF({orientation:'landscape', unit:'mm', format:'a4'});

  const pageW = pdf.internal.pageSize.getWidth();
  const pageH = pdf.internal.pageSize.getHeight();
  const left = 12, right = 12, top = 14, gap = 6;

  pdf.setFont('helvetica','bold'); pdf.setFontSize(14);
  pdf.text(`Impact of Sponsorship — ${brand}`, left, top);
  pdf.setFont('helvetica','normal'); pdf.setFontSize(10);
  pdf.text(`${event} • Generated from bls_scores`, left, top+6);

  const canvases = [
    document.getElementById('chartGrouped'),
    document.getElementById('chartFunnel'),
    document.getElementById('chartDelta')
  ];

  let y = top + 12;
  for (const c of canvases){
    const img = c.toDataURL('image/png', 1.0);
    const maxW = pageW - left - right;
    const imgW = maxW;
    const imgH = (c.height / c.width) * imgW;
    if (y + imgH > pageH - 12) { pdf.addPage(); y = top; }
    pdf.addImage(img, 'PNG', left, y, imgW, imgH);
    y += imgH + gap;
  }
  pdf.save(`bls_${event}_${brand}.pdf`);
}

/* ---------- events ---------- */
document.addEventListener('DOMContentLoaded', async ()=>{
  try {
    await loadEvents();
    await loadBrands();
    await loadCharts();
  } catch(e){
    alert('Failed to load data. Check api.php / DB connection.');
    console.error(e);
  }
});
eventSel.addEventListener('change', async ()=>{
  setParam('event', eventSel.value);
  badgeEvent.textContent = eventSel.value;
  phEvent.textContent = eventSel.value;
  await loadBrands();
  await loadCharts();
});
brandSel.addEventListener('change', loadCharts);
refreshBtn.addEventListener('click', loadCharts);

/* CSV download */
downloadBtn.addEventListener('click', ()=>{
  const event = eventSel.value;
  const brand = brandSel.value;
  fetch('./api.php?action=data&brand='+encodeURIComponent(brand)+'&event='+encodeURIComponent(event))
    .then(r=>r.json()).then(d=>{
      const rows = d.rows || [];
      const header = ['Metric','Control','Exposed_Pre','Exposed_Post','Delta_Post_vs_Pre'];
      const csv = [header.join(',')].concat(rows.map(r=>{
        const m = (r.Metric??'').replaceAll('"','""');
        const pre = num(r.Exposed_Pre), post = num(r.Exposed_Post);
        const ctrl = num(r.Control);
        const factor = (Math.max(pre||0, post||0, ctrl||0) <= 1.5) ? 100 : 1;
        const pp = (pre!=null && post!=null) ? ((post-pre)*factor).toFixed(2) : '';
        return [`"${m}"`, r.Control ?? '', r.Exposed_Pre ?? '', r.Exposed_Post ?? '', pp].join(',');
      })).join('\n');
      const blob = new Blob([csv], {type:'text/csv;charset=utf-8;'});
      const a = document.createElement('a');
      a.href = URL.createObjectURL(blob);
      a.download = `bls_${event}_${brand}.csv`;
      a.click();
    });
});
exportPngBtn.addEventListener('click', exportPNG);
exportPdfBtn.addEventListener('click', exportPDF);
printBtn.addEventListener('click', ()=> window.print());

/* re-render on resize/orientation change */
let _rt = null;
window.addEventListener('resize', ()=>{ clearTimeout(_rt); _rt = setTimeout(loadCharts, 150); });
</script>
</body>
</html>
