<?php
// /var/www/html/bls/chat.php
// Conversational "chat with your data" backend for BLS dashboard.

header('Content-Type: application/json; charset=utf-8');
session_start();
require __DIR__ . '/config.php';

const USE_OPENAI     = false;
const OLLAMA_HOST    = 'http://127.0.0.1:11434';
const OLLAMA_MODEL   = 'gemma3:latest';
const OPENAI_API_KEY = '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';
const OPENAI_MODEL   = 'gpt-4o-mini';

// ---------- helpers ----------
function out($arr){ 
    echo json_encode($arr, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); 
    exit; 
}

function http_post_json($url, $payload, $headers = []){
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => array_merge(['Content-Type: application/json'], $headers),
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_TIMEOUT => 60,
        CURLOPT_SSL_VERIFYPEER => false
    ]);
    $res = curl_exec($ch);
    $err = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return [$res, $err, $code];
}

function extract_json_from_text($text){
    $json = json_decode($text, true);
    if (is_array($json)) return $json;
    
    if (preg_match('/```(?:json)?\s*(.*?)\s*```/s', $text, $matches)) {
        $json = json_decode($matches[1], true);
        if (is_array($json)) return $json;
    }
    
    if (preg_match('/\{.*\}/s', $text, $matches)) {
        $json = json_decode($matches[0], true);
        if (is_array($json)) return $json;
    }
    
    return null;
}

// ---------- inputs ----------
$input = file_get_contents('php://input');
$data = json_decode($input, true) ?: [];

$q = trim($data['q'] ?? $_GET['q'] ?? $_POST['q'] ?? '');
$event = trim($data['event'] ?? $_GET['event'] ?? $_POST['event'] ?? '');
$brand = trim($data['brand'] ?? $_GET['brand'] ?? $_POST['brand'] ?? '');
$reset = isset($data['reset']) || isset($_GET['reset']);

if ($reset) {
    $_SESSION['chat_history'] = [];
    out(['ok' => true, 'message' => 'Chat history cleared']);
}

if ($q === '') out(['error'=>'Missing query']);

// Initialize chat history
if (!isset($_SESSION['chat_history'])) {
    $_SESSION['chat_history'] = [];
}

try {
    $pdo = db();
} catch (Throwable $e){
    out(['error'=>'DB connect failed: '.$e->getMessage()]);
}

// ---------- Build conversation history ----------
$conversationHistory = [];
foreach ($_SESSION['chat_history'] as $msg) {
    $conversationHistory[] = $msg;
}

// ---------- context ----------
$ctx = [];
if ($event !== '') $ctx[] = "Event filter: {$event}";
if ($brand !== '') $ctx[] = "Brand filter: {$brand}";
$ctxLine = $ctx ? implode("; ", $ctx) : "No filters applied";

// ---------- Enhanced conversational prompt ----------
$system_prompt = <<<PROMPT
You are a friendly and helpful data analyst assistant for Brand Lift Study (BLS) data. You chat naturally with users and help them understand their data.

DATABASE INFO:
Table: bls_scores
Columns: Event, BRAND, Metric, Control, Exposed_Pre, Exposed_Post, Sample_Size, Target_Group

CURRENT CONTEXT: $ctxLine

YOUR CAPABILITIES:
1. Answer questions about the data (query database when needed)
2. Chat naturally and remember conversation context
3. Explain metrics and insights
4. Compare brands, events, or metrics
5. Provide recommendations and analysis

RESPONSE FORMAT (JSON):
{
  "needs_data": true/false,
  "sql": "SELECT query (only if needs_data is true)",
  "answer": "Your natural, conversational response"
}

DECISION LOGIC:
- needs_data = true: When user asks about specific data, comparisons, or metrics
- needs_data = false: For greetings, clarifications, explanations, follow-ups on previous data

EXAMPLES:
User: "Hi, what can you help me with?"
Response: {"needs_data": false, "answer": "Hi! I'm your BLS data analyst. I can help you explore brand lift metrics, compare campaign performance, analyze different events, and answer questions about your data. What would you like to know?"}

User: "Show me Nike's performance"
Response: {"needs_data": true, "sql": "SELECT * FROM bls_scores WHERE BRAND = 'Nike'", "answer": "Let me pull Nike's performance data for you..."}

User: "What does that mean?"
Response: {"needs_data": false, "answer": "Based on the data I just showed you, the 'Lift' represents the percentage point increase in awareness/consideration after exposure to the campaign..."}

RULES:
- Be conversational and friendly
- Remember context from previous messages
- Generate safe SELECT queries only
- Use LIMIT 50 for performance
- Handle percentages properly (values < 2 are likely decimals, multiply by 100)
- Return ONLY valid JSON

User question: $q
PROMPT;

// ---------- call model ----------
$modelResponse = callLLM($system_prompt, $conversationHistory);

if (!$modelResponse) out(['error'=>'Empty model response']);

// ---------- parse JSON ----------
$parsed = extract_json_from_text($modelResponse);

if (!is_array($parsed)) {
    out(['error'=>'Could not parse JSON from model', 'raw_response'=>$modelResponse]);
}

$needsData = $parsed['needs_data'] ?? true;
$sql = trim($parsed['sql'] ?? '');
$answer = trim($parsed['answer'] ?? 'I apologize, I had trouble understanding that. Could you rephrase?');

$rows = [];
$usedSql = null;

// ---------- execute query if needed ----------
if ($needsData && $sql !== '') {
    // Safety checks
    if (preg_match('/\b(UPDATE|DELETE|INSERT|DROP|TRUNCATE|ALTER|CREATE|RENAME|GRANT|REVOKE)\b/i', $sql)){
        out(['error'=>'Unsafe SQL blocked']);
    }
    if (!preg_match('/^\s*SELECT\b/i', $sql)) {
        $needsData = false;
    } else {
        if (substr_count($sql, ';') > 1) out(['error'=>'Multiple statements not allowed']);
        
        $sql = rtrim($sql, ';');
        if (!preg_match('/\bLIMIT\s+\d+/i', $sql)){
            $sql .= " LIMIT 50";
        }
        
        try {
            $stmt = $pdo->query($sql);
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $usedSql = $sql;
            
            // Generate enhanced answer with actual data
            if (!empty($rows)) {
                $answer = generateConversationalAnswer($q, $rows, $answer);
            } else {
                $answer = "I looked through the data but couldn't find anything matching that criteria. Could you try being more specific or asking about a different brand/event?";
            }
            
        } catch (Throwable $e){
            $answer = "I had trouble querying that data. " . $e->getMessage();
            $needsData = false;
        }
    }
}

// ---------- Store in conversation history ----------
$_SESSION['chat_history'][] = ['role' => 'user', 'content' => $q];
$_SESSION['chat_history'][] = ['role' => 'assistant', 'content' => $answer, 'had_data' => $needsData];

// Keep only last 10 exchanges (20 messages)
if (count($_SESSION['chat_history']) > 20) {
    $_SESSION['chat_history'] = array_slice($_SESSION['chat_history'], -20);
}

out([
    'ok' => true, 
    'answer' => $answer, 
    'rows' => $rows,
    'sql' => $usedSql,
    'conversation_length' => count($_SESSION['chat_history'])
]);

// ---------- LLM caller ----------
function callLLM($prompt, $history) {
    if (!USE_OPENAI) {
        // Build full conversation for Ollama
        $fullPrompt = $prompt;
        if (!empty($history)) {
            $fullPrompt .= "\n\nPREVIOUS CONVERSATION:\n";
            foreach (array_slice($history, -6) as $msg) { // Last 3 exchanges
                $role = ucfirst($msg['role']);
                $content = $msg['content'];
                $fullPrompt .= "{$role}: {$content}\n";
            }
        }
        
        [$res, $err, $code] = http_post_json(OLLAMA_HOST.'/api/generate', [
            'model' => OLLAMA_MODEL,
            'prompt' => $fullPrompt,
            'stream' => false,
            'options' => ['temperature' => 0.3]
        ]);
        
        if ($err || $code !== 200) return null;
        
        $d = json_decode($res, true);
        return $d['response'] ?? $res;
        
    } else {
        if (!OPENAI_API_KEY) return null;
        
        $messages = [
            ['role'=>'system', 'content'=>'You are a helpful data analyst. Respond with valid JSON only.']
        ];
        
        // Add conversation history
        foreach (array_slice($history, -6) as $msg) {
            $messages[] = $msg;
        }
        
        // Add current prompt
        $messages[] = ['role'=>'user', 'content'=>$prompt];
        
        [$res, $err, $code] = http_post_json('https://api.openai.com/v1/chat/completions', [
            'model' => OPENAI_MODEL,
            'temperature' => 0.3,
            'messages' => $messages,
            'response_format' => ['type' => 'json_object']
        ], ['Authorization: Bearer '.OPENAI_API_KEY]);
        
        if ($err || $code !== 200) return null;
        
        $d = json_decode($res, true);
        return $d['choices'][0]['message']['content'] ?? '';
    }
}

// ---------- Conversational answer generator ----------
function generateConversationalAnswer($question, $rows, $baseAnswer) {
    if (empty($rows)) return $baseAnswer;
    
    $brands = array_unique(array_column($rows, 'BRAND'));
    $metrics = array_unique(array_column($rows, 'Metric'));
    $events = array_unique(array_column($rows, 'Event'));
    
    // Process data
    $processed = [];
    foreach ($rows as $row) {
        $p = $row;
        foreach (['Control', 'Exposed_Pre', 'Exposed_Post'] as $col) {
            if (isset($row[$col])) {
                $val = $row[$col];
                if (is_string($val) && strpos($val, '%') !== false) {
                    $p[$col] = floatval(str_replace('%', '', $val));
                } elseif (floatval($val) <= 1.5) {
                    $p[$col] = floatval($val) * 100;
                } else {
                    $p[$col] = floatval($val);
                }
            }
        }
        
        if (isset($p['Exposed_Pre']) && isset($p['Exposed_Post'])) {
            $p['Lift'] = round($p['Exposed_Post'] - $p['Exposed_Pre'], 2);
        }
        $processed[] = $p;
    }
    
    // Build natural insights
    $insights = [];
    
    if (count($brands) === 1) {
        $insights[] = "Looking at " . $brands[0];
    } elseif (count($brands) > 1) {
        $insights[] = "Comparing " . count($brands) . " brands: " . implode(', ', array_slice($brands, 0, 3));
    }
    
    $lifts = array_filter(array_column($processed, 'Lift'));
    if (!empty($lifts)) {
        $avgLift = round(array_sum($lifts) / count($lifts), 2);
        $maxLift = max($lifts);
        $minLift = min($lifts);
        
        if ($maxLift > 0) {
            $insights[] = "Brand lift ranges from {$minLift}% to {$maxLift}% (average: {$avgLift}%)";
        }
    }
    
    // Find top performer
    if (count($processed) > 1 && !empty($lifts)) {
        usort($processed, function($a, $b) {
            return ($b['Lift'] ?? 0) <=> ($a['Lift'] ?? 0);
        });
        $top = $processed[0];
        if (isset($top['Lift']) && $top['Lift'] > 0) {
            $insights[] = "Top performer: {$top['BRAND']} with {$top['Lift']}% lift in {$top['Metric']}";
        }
    }
    
    if (empty($insights)) {
        return $baseAnswer . " I found " . count($rows) . " records.";
    }
    
    return $baseAnswer . " " . implode('. ', $insights) . ".";
}